// Package util contaons some utility functions for the rsync poc, like bitmasking functions
package util

// SIFMT is the Filetype mask
const SIFMT = 0170000

// SIFDIR is the directory mask
const SIFDIR = 0040000

// SIFREG is the regular file mask
const SIFREG = 0100000

// SIFLNK is the symlink mask
const SIFLNK = 0120000

// IsDir returns true if the given mode is a directory
func IsDir(mode uint32) bool {
	return (mode & SIFMT) == SIFDIR
}

// IsReg returns true if the given mode is a regular file
func IsReg(mode uint32) bool {
	return (mode & SIFMT) == SIFREG
}

// IsSymlink returns true if the given mode is a symlink
func IsSymlink(mode uint32) bool {
	return (mode & SIFMT) == SIFLNK
}

const charOffset = 0

// Adler32 returns the adler32 checksum of the given data
// it seems like rsync's adler32 differs from Go's adler32.Checksum
// that's why we copy
// https://github.com/RsyncProject/rsync/blob/9615a2492bbf96bc145e738ebff55bbb91e0bbee/checksum.c#L285-L286
func Adler32(data []byte) uint32 {
	i := 0
	var s1 uint32
	var s2 uint32
	for i = 0; i < len(data)-4; i += 4 {
		s2 += 4*(s1+uint32(data[i])) + (3*uint32(data[i+1]) + 2*uint32(data[i+2]) + uint32(data[i+3]) + 10*charOffset)
		s1 += uint32(data[i]) + uint32(data[i+1]) + uint32(data[i+2]) + uint32(data[i+3]) + 4*charOffset
	}

	for ; i < len(data); i++ {
		s1 += uint32(data[i]) + charOffset
		s2 += s1
	}

	return (s1 & 0xffff) + (s2 << 16)
}
